#!/usr/bin/env python3
# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
import os
import pickle
import random
import sys

import numpy as np
import slowfast.utils.logging as logging
import torch
import torch.utils.data
from slowfast.utils.env import pathmgr
from torchvision import transforms

from . import decoder as decoder
from . import utils as utils
from . import video_container as container
from .build import DATASET_REGISTRY
from .random_erasing import RandomErasing
from .transform import create_random_augment

logger = logging.get_logger(__name__)


@DATASET_REGISTRY.register()
class Kinetics(torch.utils.data.Dataset):
    """
    Kinetics video loader. Construct the Kinetics video loader, then sample
    clips from the videos. For training and validation, a single clip is
    randomly sampled from every video with random cropping, scaling, and
    flipping. For testing, multiple clips are uniformaly sampled from every
    video with uniform cropping. For uniform cropping, we take the left, center,
    and right crop if the width is larger than height, or take top, center, and
    bottom crop if the height is larger than the width.
    """

    def __init__(self, cfg, mode, num_retries=10):
        """
        Construct the Kinetics video loader with a given csv file. The format of
        the csv file is:
        ```
        path_to_video_1 label_1
        path_to_video_2 label_2
        ...
        path_to_video_N label_N
        ```
        Args:
            cfg (CfgNode): configs.
            mode (string): Options includes `train`, `val`, or `test` mode.
                For the train and val mode, the data loader will take data
                from the train or val set, and sample one clip per video.
                For the test mode, the data loader will take data from test set,
                and sample multiple clips per video.
            num_retries (int): number of retries.
        """
        # Only support train, val, and test mode.
        assert mode in [
            "train",
            "val",
            "test",
        ], "Split '{}' not supported for Kinetics".format(mode)
        self.mode = mode
        self.cfg = cfg

        self._video_meta = {}
        self._num_retries = num_retries
        # For training or validation mode, one single clip is sampled from every
        # video. For testing, NUM_ENSEMBLE_VIEWS clips are sampled from every
        # video. For every clip, NUM_SPATIAL_CROPS is cropped spatially from
        # the frames.
        if self.mode in ["train", "val"]:
            self._num_clips = 1
        elif self.mode in ["test"]:
            self._num_clips = (
                cfg.TEST.NUM_ENSEMBLE_VIEWS * cfg.TEST.NUM_SPATIAL_CROPS
            )

        logger.info("Constructing Kinetics {}...".format(mode))
        self._construct_loader()
        self.aug = False
        self.rand_erase = False
        self.use_temporal_gradient = False
        self.temporal_gradient_rate = 0.0

        if self.mode == "train" and self.cfg.AUG.ENABLE:
            self.aug = True
            if self.cfg.AUG.RE_PROB > 0:
                self.rand_erase = True


    def _construct_loader(self):
        """
        Construct the video loader.
        """
        tmp = "new_{}.list".format(self.mode)
        # root = self.cfg.DATA.PATH_TO_DATA_DIR
        root = ""

        path_to_file = os.path.join(root, tmp)

        assert pathmgr.exists(path_to_file), "{} dir not found".format(
            path_to_file
        )

        with open(path_to_file, "rb") as f:
            path_list = pickle.load(f)

        self._path_to_videos = []
        self._path_to_audios = []

        self._labels = []
        self._spatial_temporal_idx = []

        for clip_idx, (video_path, audio_path, label) in enumerate(path_list):

            for idx in range(self._num_clips):
                self._path_to_videos.append(video_path)
                self._path_to_audios.append(audio_path)
                self._labels.append(int(label))

                self._spatial_temporal_idx.append(idx)
                self._video_meta[clip_idx * self._num_clips + idx] = {}

        assert (
            len(self._path_to_videos) > 0
        ), "Failed to load Kinetics split {} from {}".format(
            self._split_idx, path_to_file
        )
        logger.info(
            "Constructing kinetics dataloader (size: {}) from {}".format(
                len(self._path_to_videos), path_to_file
            )
        )


    #
    # def _construct_loader(self):
    #     """
    #     Construct the video loader.
    #     """
    #     path_to_file = os.path.join(
    #         self.cfg.DATA.PATH_TO_DATA_DIR, "{}.csv".format(self.mode)
    #     )
    #     assert pathmgr.exists(path_to_file), "{} dir not found".format(
    #         path_to_file
    #     )
    #
    #     self._path_to_videos = []
    #     self._labels = []
    #     self._spatial_temporal_idx = []
    #     with pathmgr.open(path_to_file, "r") as f:
    #         for clip_idx, path_label in enumerate(f.read().splitlines()):
    #             path_label = path_label[1: len(path_label)-1]
    #
    #             assert (
    #                 len(path_label.split(self.cfg.DATA.PATH_LABEL_SEPARATOR))
    #                 == 2
    #             )
    #             path, label = path_label.split(
    #                 self.cfg.DATA.PATH_LABEL_SEPARATOR
    #             )
    #             for idx in range(self._num_clips):
    #                 self._path_to_videos.append(
    #                     os.path.join(self.cfg.DATA.PATH_PREFIX, path)
    #                 )
    #                 self._labels.append(int(label))
    #                 self._spatial_temporal_idx.append(idx)
    #                 self._video_meta[clip_idx * self._num_clips + idx] = {}
    #     assert (
    #         len(self._path_to_videos) > 0
    #     ), "Failed to load Kinetics split {} from {}".format(
    #         self._split_idx, path_to_file
    #     )
    #     logger.info(
    #         "Constructing kinetics dataloader (size: {}) from {}".format(
    #             len(self._path_to_videos), path_to_file
    #         )
    #     )

    def __getitem__(self, index):
        """
        Given the video index, return the list of frames, label, and video
        index if the video can be fetched and decoded successfully, otherwise
        repeatly find a random video that can be decoded as a replacement.
        Args:
            index (int): the video index provided by the pytorch sampler.
        Returns:
            frames (tensor): the frames of sampled from the video. The dimension
                is `channel` x `num frames` x `height` x `width`.
            label (int): the label of the current video.
            index (int): if the video provided by pytorch sampler can be
                decoded, then return the index of the video. If not, return the
                index of the video replacement that can be decoded.
        """
        short_cycle_idx = None

        # When short cycle is used, input index is a tupple.
        if isinstance(index, tuple):
            index, short_cycle_idx = index

        if self.mode in ["train", "val"]:
            # -1 indicates random sampling. short_cycle_idx=None, index=int
            temporal_sample_index = -1
            spatial_sample_index = -1
            min_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[0]
            max_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[1]
            crop_size = self.cfg.DATA.TRAIN_CROP_SIZE
            if short_cycle_idx in [0, 1]:
                crop_size = int(
                    round(
                        self.cfg.MULTIGRID.SHORT_CYCLE_FACTORS[short_cycle_idx]
                        * self.cfg.MULTIGRID.DEFAULT_S
                    )
                )

            if self.cfg.MULTIGRID.DEFAULT_S > 0: # equal to 0
                # Decreasing the scale is equivalent to using a larger "span"
                # in a sampling grid.
                min_scale = int(
                    round(
                        float(min_scale)
                        * crop_size
                        / self.cfg.MULTIGRID.DEFAULT_S
                    )
                )
        elif self.mode in ["test"]:
            temporal_sample_index = (
                self._spatial_temporal_idx[index]
                // self.cfg.TEST.NUM_SPATIAL_CROPS
            )
            # spatial_sample_index is in [0, 1, 2]. Corresponding to left,
            # center, or right if width is larger than height, and top, middle,
            # or bottom if height is larger than width.
            spatial_sample_index = (
                (
                    self._spatial_temporal_idx[index]
                    % self.cfg.TEST.NUM_SPATIAL_CROPS
                )
                if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
                else 1
            )
            min_scale, max_scale, crop_size = (
                [self.cfg.DATA.TEST_CROP_SIZE] * 3
                if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
                else [self.cfg.DATA.TRAIN_JITTER_SCALES[0]] * 2
                + [self.cfg.DATA.TEST_CROP_SIZE]
            )
            # The testing is deterministic and no jitter should be performed.
            # min_scale, max_scale, and crop_size are expect to be the same.
            assert len({min_scale, max_scale}) == 1
        else:
            raise NotImplementedError(
                "Does not support {} mode".format(self.mode)
            )
        sampling_rate = utils.get_random_sampling_rate(
            self.cfg.MULTIGRID.LONG_CYCLE_SAMPLING_RATE, # 0
            self.cfg.DATA.SAMPLING_RATE,
        )
        # Try to decode and sample a clip from a video. If the video can not be
        # decoded, repeatly find a random video replacement that can be decoded.
        for i_try in range(self._num_retries):
            video_container = None
            try:
                video_container = container.get_video_container( #av.open(0
                    self._path_to_videos[index],
                    self.cfg.DATA_LOADER.ENABLE_MULTI_THREAD_DECODE,
                    self.cfg.DATA.DECODING_BACKEND,
                )
            except Exception as e:
                logger.info(
                    "Failed to load video from {} with error {}".format(
                        self._path_to_videos[index], e
                    )
                )
            # Select a random video if the current video was not able to access.
            if video_container is None:
                logger.warning(
                    "Failed to meta load video idx {} from {}; trial {}".format(
                        index, self._path_to_videos[index], i_try
                    )
                )
                if self.mode not in ["test"] and i_try > self._num_retries // 2:
                    # let's try another one
                    index = random.randint(0, len(self._path_to_videos) - 1)
                continue

            # Decode video. Meta info is used to perform selective decoding.
            frames = decoder.decode(
                video_container,
                sampling_rate,
                self.cfg.DATA.NUM_FRAMES,
                temporal_sample_index,
                self.cfg.TEST.NUM_ENSEMBLE_VIEWS,
                video_meta=self._video_meta[index],
                target_fps=self.cfg.DATA.TARGET_FPS,
                backend=self.cfg.DATA.DECODING_BACKEND,
                max_spatial_scale=min_scale,
                use_offset=self.cfg.DATA.USE_OFFSET_SAMPLING,
            )

            # If decoding failed (wrong format, video is too short, and etc),
            # select another video.
            if frames is None:
                logger.warning(
                    "Failed to decode video idx {} from {}; trial {}".format(
                        index, self._path_to_videos[index], i_try
                    )
                )
                if self.mode not in ["test"] and i_try > self._num_retries // 2:
                    # let's try another one
                    index = random.randint(0, len(self._path_to_videos) - 1)
                continue

            if self.aug:
                if self.cfg.AUG.NUM_SAMPLE > 1:

                    frame_list = []
                    label_list = []
                    index_list = []
                    for _ in range(self.cfg.AUG.NUM_SAMPLE):
                        new_frames = self._aug_frame(
                            frames,
                            spatial_sample_index,
                            min_scale,
                            max_scale,
                            crop_size,
                        )
                        label = self._labels[index]
                        new_frames = utils.pack_pathway_output(
                            self.cfg, new_frames
                        )
                        frame_list.append(new_frames)
                        label_list.append(label)
                        index_list.append(index)
                    return frame_list, label_list, index_list, {}

                else:
                    frames = self._aug_frame(
                        frames,
                        spatial_sample_index,
                        min_scale,
                        max_scale,
                        crop_size,
                    )

            else:
                frames = utils.tensor_normalize(
                    frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
                )
                # T H W C -> C T H W.
                frames = frames.permute(3, 0, 1, 2)
                # Perform data augmentation.
                frames = utils.spatial_sampling(
                    frames,
                    spatial_idx=spatial_sample_index,
                    min_scale=min_scale,
                    max_scale=max_scale,
                    crop_size=crop_size,
                    random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
                    inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
                )
            label = self._labels[index]
            frames = utils.pack_pathway_output(self.cfg, frames)
            return frames, label, index, {}
        else:
            raise RuntimeError(
                "Failed to fetch video after {} retries.".format(
                    self._num_retries
                )
            )

    def _aug_frame(
        self,
        frames,
        spatial_sample_index,
        min_scale,
        max_scale,
        crop_size,
    ):
        aug_transform = create_random_augment(
            input_size=(frames.size(1), frames.size(2)),
            auto_augment=self.cfg.AUG.AA_TYPE,
            interpolation=self.cfg.AUG.INTERPOLATION,
        )
        # T H W C -> T C H W.
        frames = frames.permute(0, 3, 1, 2)
        list_img = self._frame_to_list_img(frames)
        list_img = aug_transform(list_img)
        frames = self._list_img_to_frames(list_img)
        frames = frames.permute(0, 2, 3, 1)

        frames = utils.tensor_normalize(
            frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
        )
        # T H W C -> C T H W.
        frames = frames.permute(3, 0, 1, 2)
        # Perform data augmentation.
        scl, asp = (
            self.cfg.DATA.TRAIN_JITTER_SCALES_RELATIVE,
            self.cfg.DATA.TRAIN_JITTER_ASPECT_RELATIVE,
        )
        relative_scales = (
            None if (self.mode not in ["train"] or len(scl) == 0) else scl
        )
        relative_aspect = (
            None if (self.mode not in ["train"] or len(asp) == 0) else asp
        )
        frames = utils.spatial_sampling(
            frames,
            spatial_idx=spatial_sample_index,
            min_scale=min_scale,
            max_scale=max_scale,
            crop_size=crop_size,
            random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
            inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
            aspect_ratio=relative_aspect,
            scale=relative_scales,
            motion_shift=self.cfg.DATA.TRAIN_JITTER_MOTION_SHIFT
            if self.mode in ["train"]
            else False,
        )

        if self.rand_erase:
            erase_transform = RandomErasing(
                self.cfg.AUG.RE_PROB,
                mode=self.cfg.AUG.RE_MODE,
                max_count=self.cfg.AUG.RE_COUNT,
                num_splits=self.cfg.AUG.RE_COUNT,
                device="cpu",
            )
            frames = frames.permute(1, 0, 2, 3)
            frames = erase_transform(frames)
            frames = frames.permute(1, 0, 2, 3)

        return frames

    def _frame_to_list_img(self, frames):
        img_list = [
            transforms.ToPILImage()(frames[i]) for i in range(frames.size(0))
        ]
        return img_list

    def _list_img_to_frames(self, img_list):
        img_list = [transforms.ToTensor()(img) for img in img_list]
        return torch.stack(img_list)

    def __len__(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return self.num_videos

    @property
    def num_videos(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return len(self._path_to_videos)















# @DATASET_REGISTRY.register()
# class Kinetics_av(torch.utils.data.Dataset):
#     """
#     Kinetics video loader. Construct the Kinetics video loader, then sample
#     clips from the videos. For training and validation, a single clip is
#     randomly sampled from every video with random cropping, scaling, and
#     flipping. For testing, multiple clips are uniformaly sampled from every
#     video with uniform cropping. For uniform cropping, we take the left, center,
#     and right crop if the width is larger than height, or take top, center, and
#     bottom crop if the height is larger than the width.
#     """
#
#     def __init__(self, cfg, mode, num_retries=10):
#         """
#         Construct the Kinetics video loader with a given csv file. The format of
#         the csv file is:
#         ```
#         path_to_video_1 label_1
#         path_to_video_2 label_2
#         ...
#         path_to_video_N label_N
#         ```
#         Args:
#             cfg (CfgNode): configs.
#             mode (string): Options includes `train`, `val`, or `test` mode.
#                 For the train and val mode, the data loader will take data
#                 from the train or val set, and sample one clip per video.
#                 For the test mode, the data loader will take data from test set,
#                 and sample multiple clips per video.
#             num_retries (int): number of retries.
#         """
#         # Only support train, val, and test mode.
#         assert mode in [
#             "train",
#             "val",
#             "test",
#         ], "Split '{}' not supported for Kinetics".format(mode)
#         self.mode = mode
#         self.cfg = cfg
#         self.stft = STFT(n_fft=1024, win_length=1024, hop_length=512, sr=16000, output_format='Magnitude')
#
#         self._video_meta = {}
#         self._num_retries = num_retries
#         # For training or validation mode, one single clip is sampled from every
#         # video. For testing, NUM_ENSEMBLE_VIEWS clips are sampled from every
#         # video. For every clip, NUM_SPATIAL_CROPS is cropped spatially from
#         # the frames.
#         if self.mode in ["train", "val"]:
#             self._num_clips = 1
#         elif self.mode in ["test"]:
#             self._num_clips = (
#                 cfg.TEST.NUM_ENSEMBLE_VIEWS * cfg.TEST.NUM_SPATIAL_CROPS
#             )
#
#         logger.info("Constructing Kinetics {}...".format(mode))
#         self._construct_loader()
#         self.aug = False
#         self.rand_erase = False
#         self.use_temporal_gradient = False
#         self.temporal_gradient_rate = 0.0
#
#         if self.mode == "train" and self.cfg.AUG.ENABLE:
#             self.aug = True
#             if self.cfg.AUG.RE_PROB > 0:
#                 self.rand_erase = True
#
#
#     def _construct_loader(self):
#         """
#         Construct the video loader.
#         """
#         tmp = "new_{}.list".format(self.mode)
#         # root = self.cfg.DATA.PATH_TO_DATA_DIR
#
#         path_to_file = os.path.join(root, tmp)
#
#         assert pathmgr.exists(path_to_file), "{} dir not found".format(
#             path_to_file
#         )
#
#         with open(path_to_file, "rb") as f:
#             path_list = pickle.load(f)
#
#         self._path_to_videos = []
#         self._path_to_audios = []
#
#         self._labels = []
#         self._spatial_temporal_idx = []
#
#         for clip_idx, (video_path, audio_path, label) in enumerate(path_list):
#
#             for idx in range(self._num_clips):
#                 self._path_to_videos.append(video_path)
#                 self._path_to_audios.append(audio_path)
#                 self._labels.append(int(label))
#
#                 self._spatial_temporal_idx.append(idx)
#                 self._video_meta[clip_idx * self._num_clips + idx] = {}
#
#         assert (
#             len(self._path_to_videos) > 0
#         ), "Failed to load Kinetics split {} from {}".format(
#             self._split_idx, path_to_file
#         )
#         logger.info(
#             "Constructing kinetics dataloader (size: {}) from {}".format(
#                 len(self._path_to_videos), path_to_file
#             )
#         )
#
#     def read_wave(self, wav_path, sr=16000, mono=True):
#         try:
#             wav, _ = librosa.load(wav_path, sr=sr, mono=mono)
#         except:
#             raise AssertionError("Unsupported file: %s" % wav_path)
#         return wav
#
#     def turn_wav(self, wav,  frames=int(160000)):
#         if wav.shape[0] <= frames:
#             len_pad = frames - wav.shape[0]
#             new_wav = np.pad(wav, ((0, len_pad)), 'constant', constant_values=(0.,))
#
#         else:
#             if self.mode == "train":
#                 start = random.choice((range(0, wav.shape[0] - frames)))
#             else:
#                 start = 0
#             new_wav = wav[start: start + frames]
#
#         return new_wav
#
#     def wave_normalize(self, wav):
#         norm = np.max(np.abs(wav)) * 1.1
#         wav = wav / norm
#         return wav
#
#     def get_audio_input(self, audio_path):
#         wav = self.read_wave(audio_path)
#         wav = self.turn_wav(wav)
#         wav = self.wave_normalize(wav)
#         wav = torch.from_numpy(wav)
#         stft = self.stft(wav)
#         return stft
#
#     def __getitem__(self, index):
#         """
#         Given the video index, return the list of frames, label, and video
#         index if the video can be fetched and decoded successfully, otherwise
#         repeatly find a random video that can be decoded as a replacement.
#         Args:
#             index (int): the video index provided by the pytorch sampler.
#         Returns:
#             frames (tensor): the frames of sampled from the video. The dimension
#                 is `channel` x `num frames` x `height` x `width`.
#             label (int): the label of the current video.
#             index (int): if the video provided by pytorch sampler can be
#                 decoded, then return the index of the video. If not, return the
#                 index of the video replacement that can be decoded.
#         """
#
#         short_cycle_idx = None
#         audio = self.get_audio_input(self._path_to_audios[index])
#
#         # When short cycle is used, input index is a tupple.
#         if isinstance(index, tuple):
#             index, short_cycle_idx = index
#
#         if self.mode in ["train", "val"]:
#             # -1 indicates random sampling.
#             temporal_sample_index = -1
#             spatial_sample_index = -1
#             min_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[0]
#             max_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[1]
#             crop_size = self.cfg.DATA.TRAIN_CROP_SIZE
#             if short_cycle_idx in [0, 1]:
#                 crop_size = int(
#                     round(
#                         self.cfg.MULTIGRID.SHORT_CYCLE_FACTORS[short_cycle_idx]
#                         * self.cfg.MULTIGRID.DEFAULT_S
#                     )
#                 )
#
#             if self.cfg.MULTIGRID.DEFAULT_S > 0:
#                 # Decreasing the scale is equivalent to using a larger "span"
#                 # in a sampling grid.
#                 min_scale = int(
#                     round(
#                         float(min_scale)
#                         * crop_size
#                         / self.cfg.MULTIGRID.DEFAULT_S
#                     )
#                 )
#         elif self.mode in ["test"]:
#             temporal_sample_index = (
#                 self._spatial_temporal_idx[index]
#                 // self.cfg.TEST.NUM_SPATIAL_CROPS
#             )
#             # spatial_sample_index is in [0, 1, 2]. Corresponding to left,
#             # center, or right if width is larger than height, and top, middle,
#             # or bottom if height is larger than width.
#             spatial_sample_index = (
#                 (
#                     self._spatial_temporal_idx[index]
#                     % self.cfg.TEST.NUM_SPATIAL_CROPS
#                 )
#                 if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
#                 else 1
#             )
#             min_scale, max_scale, crop_size = (
#                 [self.cfg.DATA.TEST_CROP_SIZE] * 3
#                 if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
#                 else [self.cfg.DATA.TRAIN_JITTER_SCALES[0]] * 2
#                 + [self.cfg.DATA.TEST_CROP_SIZE]
#             )
#             # The testing is deterministic and no jitter should be performed.
#             # min_scale, max_scale, and crop_size are expect to be the same.
#             assert len({min_scale, max_scale}) == 1
#         else:
#             raise NotImplementedError(
#                 "Does not support {} mode".format(self.mode)
#             )
#         sampling_rate = utils.get_random_sampling_rate(
#             self.cfg.MULTIGRID.LONG_CYCLE_SAMPLING_RATE,
#             self.cfg.DATA.SAMPLING_RATE,
#         )
#         # Try to decode and sample a clip from a video. If the video can not be
#         # decoded, repeatly find a random video replacement that can be decoded.
#         for i_try in range(self._num_retries):
#             video_container = None
#             try:
#                 video_container = container.get_video_container(
#                     self._path_to_videos[index],
#                     self.cfg.DATA_LOADER.ENABLE_MULTI_THREAD_DECODE,
#                     self.cfg.DATA.DECODING_BACKEND,
#                 )
#             except Exception as e:
#                 logger.info(
#                     "Failed to load video from {} with error {}".format(
#                         self._path_to_videos[index], e
#                     )
#                 )
#             # Select a random video if the current video was not able to access.
#             if video_container is None:
#                 logger.warning(
#                     "Failed to meta load video idx {} from {}; trial {}".format(
#                         index, self._path_to_videos[index], i_try
#                     )
#                 )
#                 if self.mode not in ["test"] and i_try > self._num_retries // 2:
#                     # let's try another one
#                     index = random.randint(0, len(self._path_to_videos) - 1)
#                 continue
#
#             # Decode video. Meta info is used to perform selective decoding.
#             frames = decoder.decode(
#                 video_container,
#                 sampling_rate,
#                 self.cfg.DATA.NUM_FRAMES,
#                 temporal_sample_index,
#                 self.cfg.TEST.NUM_ENSEMBLE_VIEWS,
#                 video_meta=self._video_meta[index],
#                 target_fps=self.cfg.DATA.TARGET_FPS,
#                 backend=self.cfg.DATA.DECODING_BACKEND,
#                 max_spatial_scale=min_scale,
#                 use_offset=self.cfg.DATA.USE_OFFSET_SAMPLING,
#             )
#
#             # If decoding failed (wrong format, video is too short, and etc),
#             # select another video.
#             if frames is None:
#                 logger.warning(
#                     "Failed to decode video idx {} from {}; trial {}".format(
#                         index, self._path_to_videos[index], i_try
#                     )
#                 )
#                 if self.mode not in ["test"] and i_try > self._num_retries // 2:
#                     # let's try another one
#                     index = random.randint(0, len(self._path_to_videos) - 1)
#                 continue
#
#             if self.aug:
#                 if self.cfg.AUG.NUM_SAMPLE > 1:
#
#                     frame_list = []
#                     label_list = []
#                     index_list = []
#                     for _ in range(self.cfg.AUG.NUM_SAMPLE):
#                         new_frames = self._aug_frame(
#                             frames,
#                             spatial_sample_index,
#                             min_scale,
#                             max_scale,
#                             crop_size,
#                         )
#                         label = self._labels[index]
#                         new_frames = utils.pack_pathway_output(
#                             self.cfg, new_frames
#                         )
#                         frame_list.append(new_frames)
#                         label_list.append(label)
#                         index_list.append(index)
#                     return frame_list, label_list, index_list, {}
#
#                 else:
#                     frames = self._aug_frame(
#                         frames,
#                         spatial_sample_index,
#                         min_scale,
#                         max_scale,
#                         crop_size,
#                     )
#
#             else:
#                 frames = utils.tensor_normalize(
#                     frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
#                 )
#                 # T H W C -> C T H W.
#                 frames = frames.permute(3, 0, 1, 2)
#                 # Perform data augmentation.
#                 frames = utils.spatial_sampling(
#                     frames,
#                     spatial_idx=spatial_sample_index,
#                     min_scale=min_scale,
#                     max_scale=max_scale,
#                     crop_size=crop_size,
#                     random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
#                     inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
#                 )
#             label = self._labels[index]
#             frames = utils.pack_pathway_output(self.cfg, frames)
#
#             return audio, frames, label, index, {}
#         else:
#             raise RuntimeError(
#                 "Failed to fetch video after {} retries.".format(
#                     self._num_retries
#                 )
#             )
#
#     def _aug_frame(
#         self,
#         frames,
#         spatial_sample_index,
#         min_scale,
#         max_scale,
#         crop_size,
#     ):
#         aug_transform = create_random_augment(
#             input_size=(frames.size(1), frames.size(2)),
#             auto_augment=self.cfg.AUG.AA_TYPE,
#             interpolation=self.cfg.AUG.INTERPOLATION,
#         )
#         # T H W C -> T C H W.
#         frames = frames.permute(0, 3, 1, 2)
#         list_img = self._frame_to_list_img(frames)
#         list_img = aug_transform(list_img)
#         frames = self._list_img_to_frames(list_img)
#         frames = frames.permute(0, 2, 3, 1)
#
#         frames = utils.tensor_normalize(
#             frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
#         )
#         # T H W C -> C T H W.
#         frames = frames.permute(3, 0, 1, 2)
#         # Perform data augmentation.
#         scl, asp = (
#             self.cfg.DATA.TRAIN_JITTER_SCALES_RELATIVE,
#             self.cfg.DATA.TRAIN_JITTER_ASPECT_RELATIVE,
#         )
#         relative_scales = (
#             None if (self.mode not in ["train"] or len(scl) == 0) else scl
#         )
#         relative_aspect = (
#             None if (self.mode not in ["train"] or len(asp) == 0) else asp
#         )
#         frames = utils.spatial_sampling(
#             frames,
#             spatial_idx=spatial_sample_index,
#             min_scale=min_scale,
#             max_scale=max_scale,
#             crop_size=crop_size,
#             random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
#             inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
#             aspect_ratio=relative_aspect,
#             scale=relative_scales,
#             motion_shift=self.cfg.DATA.TRAIN_JITTER_MOTION_SHIFT
#             if self.mode in ["train"]
#             else False,
#         )
#
#         if self.rand_erase:
#             erase_transform = RandomErasing(
#                 self.cfg.AUG.RE_PROB,
#                 mode=self.cfg.AUG.RE_MODE,
#                 max_count=self.cfg.AUG.RE_COUNT,
#                 num_splits=self.cfg.AUG.RE_COUNT,
#                 device="cpu",
#             )
#             frames = frames.permute(1, 0, 2, 3)
#             frames = erase_transform(frames)
#             frames = frames.permute(1, 0, 2, 3)
#
#         return frames
#
#     def _frame_to_list_img(self, frames):
#         img_list = [
#             transforms.ToPILImage()(frames[i]) for i in range(frames.size(0))
#         ]
#         return img_list
#
#     def _list_img_to_frames(self, img_list):
#         img_list = [transforms.ToTensor()(img) for img in img_list]
#         return torch.stack(img_list)
#
#     def __len__(self):
#         """
#         Returns:
#             (int): the number of videos in the dataset.
#         """
#         return self.num_videos
#
#     @property
#     def num_videos(self):
#         """
#         Returns:
#             (int): the number of videos in the dataset.
#         """
#         return len(self._path_to_videos)








@DATASET_REGISTRY.register()
class Kinetics_av_align(torch.utils.data.Dataset):
    """
    Kinetics video loader. Construct the Kinetics video loader, then sample
    clips from the videos. For training and validation, a single clip is
    randomly sampled from every video with random cropping, scaling, and
    flipping. For testing, multiple clips are uniformaly sampled from every
    video with uniform cropping. For uniform cropping, we take the left, center,
    and right crop if the width is larger than height, or take top, center, and
    bottom crop if the height is larger than the width.
    """

    def __init__(self, cfg, mode, num_retries=10):
        """
        Construct the Kinetics video loader with a given csv file. The format of
        the csv file is:
        ```
        path_to_video_1 label_1
        path_to_video_2 label_2
        ...
        path_to_video_N label_N
        ```
        Args:
            cfg (CfgNode): configs.
            mode (string): Options includes `train`, `val`, or `test` mode.
                For the train and val mode, the data loader will take data
                from the train or val set, and sample one clip per video.
                For the test mode, the data loader will take data from test set,
                and sample multiple clips per video.
            num_retries (int): number of retries.
        """
        # Only support train, val, and test mode.
        assert mode in [
            "train",
            "val",
            "test",
        ], "Split '{}' not supported for Kinetics".format(mode)
        self.mode = mode
        self.cfg = cfg

        self._video_meta = {}
        self._num_retries = num_retries
        # For training or validation mode, one single clip is sampled from every
        # video. For testing, NUM_ENSEMBLE_VIEWS clips are sampled from every
        # video. For every clip, NUM_SPATIAL_CROPS is cropped spatially from
        # the frames.
        if self.mode in ["train", "val"]:
            self._num_clips = 1
        elif self.mode in ["test"]:
            self._num_clips = (
                cfg.TEST.NUM_ENSEMBLE_VIEWS * cfg.TEST.NUM_SPATIAL_CROPS
            )

        logger.info("Constructing Kinetics {}...".format(mode))
        self._construct_loader()
        self.aug = False
        self.rand_erase = False
        self.use_temporal_gradient = False
        self.temporal_gradient_rate = 0.0

        if self.mode == "train" and self.cfg.AUG.ENABLE:
            self.aug = True
            if self.cfg.AUG.RE_PROB > 0:
                self.rand_erase = True


    def _construct_loader(self):
        """
        Construct the video loader.
        """
        tmp = "kinetics400_{}_list_audio_feature.txt".format(self.mode)
        # root = self.cfg.DATA.PATH_TO_DATA_DIR
        root = ""

        path_to_file = os.path.join(root, tmp)

        assert pathmgr.exists(path_to_file), "{} dir not found".format(
            path_to_file
        )

        path_list = []
        with open(path_to_file, 'r') as lines:
            for line in lines:
                line_split = line.strip().split()
                audio_path = line_split[0]

                video_path = audio_path.replace("kinetics400audio_mel", "kinetics400video_size256")
                video_path = video_path[:len(video_path) - 4]

                assert os.path.exists(video_path), "{}".format(video_path)
                label = int(line_split[2])
                path_list.append([audio_path, video_path, label])


        self._path_to_videos = []
        self._path_to_audios = []

        self._labels = []
        self._spatial_temporal_idx = []

        for clip_idx, (audio_path, video_path, label) in enumerate(path_list):

            for idx in range(self._num_clips):
                self._path_to_videos.append(video_path)
                self._path_to_audios.append(audio_path)
                self._labels.append(int(label))

                self._spatial_temporal_idx.append(idx)
                self._video_meta[clip_idx * self._num_clips + idx] = {}

        assert (
            len(self._path_to_videos) > 0
        ), "Failed to load Kinetics split {} from {}".format(
            self._split_idx, path_to_file
        )
        logger.info(
            "Constructing kinetics dataloader (size: {}) from {}".format(
                len(self._path_to_videos), path_to_file
            )
        )


    def get_audio_input(self, audio_path, begin_frame, total_frame):
        audio = np.load(audio_path)
        begin_frame = int(begin_frame)
        begin = int(begin_frame / total_frame * audio.shape[0])
        if audio.shape[0] > begin + 128:
            cropped_audio = audio[begin: begin + 128]
        else:
            cropped_audio = audio[begin:]
            cropped_audio = np.pad(
                cropped_audio,
                ((0, 128 - cropped_audio.shape[0]), (0, 0)),
                mode='constant')

        assert cropped_audio.shape[0] == 128 and cropped_audio.shape[1] == 80
        cropped_audio = cropped_audio.reshape((1, cropped_audio.shape[0], cropped_audio.shape[1]))
        cropped_audio = torch.from_numpy(cropped_audio).float()

        return cropped_audio

    def __getitem__(self, index):
        """
        Given the video index, return the list of frames, label, and video
        index if the video can be fetched and decoded successfully, otherwise
        repeatly find a random video that can be decoded as a replacement.
        Args:
            index (int): the video index provided by the pytorch sampler.
        Returns:
            frames (tensor): the frames of sampled from the video. The dimension
                is `channel` x `num frames` x `height` x `width`.
            label (int): the label of the current video.
            index (int): if the video provided by pytorch sampler can be
                decoded, then return the index of the video. If not, return the
                index of the video replacement that can be decoded.
        """

        short_cycle_idx = None

        # When short cycle is used, input index is a tupple.
        if isinstance(index, tuple):
            index, short_cycle_idx = index

        if self.mode in ["train", "val"]:
            # -1 indicates random sampling.
            temporal_sample_index = -1
            spatial_sample_index = -1
            min_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[0]
            max_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[1]
            crop_size = self.cfg.DATA.TRAIN_CROP_SIZE
            if short_cycle_idx in [0, 1]:
                crop_size = int(
                    round(
                        self.cfg.MULTIGRID.SHORT_CYCLE_FACTORS[short_cycle_idx]
                        * self.cfg.MULTIGRID.DEFAULT_S
                    )
                )

            if self.cfg.MULTIGRID.DEFAULT_S > 0:
                # Decreasing the scale is equivalent to using a larger "span"
                # in a sampling grid.
                min_scale = int(
                    round(
                        float(min_scale)
                        * crop_size
                        / self.cfg.MULTIGRID.DEFAULT_S
                    )
                )
        elif self.mode in ["test"]:
            temporal_sample_index = (
                self._spatial_temporal_idx[index]
                // self.cfg.TEST.NUM_SPATIAL_CROPS
            )
            # spatial_sample_index is in [0, 1, 2]. Corresponding to left,
            # center, or right if width is larger than height, and top, middle,
            # or bottom if height is larger than width.
            spatial_sample_index = (
                (
                    self._spatial_temporal_idx[index]
                    % self.cfg.TEST.NUM_SPATIAL_CROPS
                )
                if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
                else 1
            )
            min_scale, max_scale, crop_size = (
                [self.cfg.DATA.TEST_CROP_SIZE] * 3
                if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
                else [self.cfg.DATA.TRAIN_JITTER_SCALES[0]] * 2
                + [self.cfg.DATA.TEST_CROP_SIZE]
            )
            # The testing is deterministic and no jitter should be performed.
            # min_scale, max_scale, and crop_size are expect to be the same.
            assert len({min_scale, max_scale}) == 1
        else:
            raise NotImplementedError(
                "Does not support {} mode".format(self.mode)
            )
        sampling_rate = utils.get_random_sampling_rate(
            self.cfg.MULTIGRID.LONG_CYCLE_SAMPLING_RATE,
            self.cfg.DATA.SAMPLING_RATE,
        )
        # Try to decode and sample a clip from a video. If the video can not be
        # decoded, repeatly find a random video replacement that can be decoded.
        for i_try in range(self._num_retries):
            video_container = None
            try:
                video_container = container.get_video_container(
                    self._path_to_videos[index],
                    self.cfg.DATA_LOADER.ENABLE_MULTI_THREAD_DECODE,
                    self.cfg.DATA.DECODING_BACKEND,
                )
            except Exception as e:
                logger.info(
                    "Failed to load video from {} with error {}".format(
                        self._path_to_videos[index], e
                    )
                )
            # Select a random video if the current video was not able to access.
            if video_container is None:
                logger.warning(
                    "Failed to meta load video idx {} from {}; trial {}".format(
                        index, self._path_to_videos[index], i_try
                    )
                )
                if self.mode not in ["test"] and i_try > self._num_retries // 2:
                    # let's try another one
                    index = random.randint(0, len(self._path_to_videos) - 1)
                continue

            # Decode video. Meta info is used to perform selective decoding.
            frames, return_start_idx, return_frame_length = decoder.decode(
                video_container,
                sampling_rate,
                self.cfg.DATA.NUM_FRAMES,
                temporal_sample_index,
                self.cfg.TEST.NUM_ENSEMBLE_VIEWS,
                video_meta=self._video_meta[index],
                target_fps=self.cfg.DATA.TARGET_FPS,
                backend=self.cfg.DATA.DECODING_BACKEND,
                max_spatial_scale=min_scale,
                use_offset=self.cfg.DATA.USE_OFFSET_SAMPLING,
                audio_visual_align=True
            )

            # If decoding failed (wrong format, video is too short, and etc),
            # select another video.
            if frames is None:
                logger.warning(
                    "Failed to decode video idx {} from {}; trial {}".format(
                        index, self._path_to_videos[index], i_try
                    )
                )
                if self.mode not in ["test"] and i_try > self._num_retries // 2:
                    # let's try another one
                    index = random.randint(0, len(self._path_to_videos) - 1)
                continue

            if self.aug:
                if self.cfg.AUG.NUM_SAMPLE > 1:

                    frame_list = []
                    label_list = []
                    index_list = []
                    for _ in range(self.cfg.AUG.NUM_SAMPLE):
                        new_frames = self._aug_frame(
                            frames,
                            spatial_sample_index,
                            min_scale,
                            max_scale,
                            crop_size,
                        )
                        label = self._labels[index]
                        new_frames = utils.pack_pathway_output(
                            self.cfg, new_frames
                        )
                        frame_list.append(new_frames)
                        label_list.append(label)
                        index_list.append(index)
                    return frame_list, label_list, index_list, {}

                else:
                    frames = self._aug_frame(
                        frames,
                        spatial_sample_index,
                        min_scale,
                        max_scale,
                        crop_size,
                    )

            else:
                frames = utils.tensor_normalize(
                    frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
                )
                # T H W C -> C T H W.
                frames = frames.permute(3, 0, 1, 2)
                # Perform data augmentation.
                frames = utils.spatial_sampling(
                    frames,
                    spatial_idx=spatial_sample_index,
                    min_scale=min_scale,
                    max_scale=max_scale,
                    crop_size=crop_size,
                    random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
                    inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
                )
            label = self._labels[index]
            frames = utils.pack_pathway_output(self.cfg, frames)

            audio = self.get_audio_input(self._path_to_audios[index], return_start_idx, return_frame_length)

            return audio, frames, label, index, {}
        else:
            raise RuntimeError(
                "Failed to fetch video after {} retries.".format(
                    self._num_retries
                )
            )

    def _aug_frame(
        self,
        frames,
        spatial_sample_index,
        min_scale,
        max_scale,
        crop_size,
    ):
        aug_transform = create_random_augment(
            input_size=(frames.size(1), frames.size(2)),
            auto_augment=self.cfg.AUG.AA_TYPE,
            interpolation=self.cfg.AUG.INTERPOLATION,
        )
        # T H W C -> T C H W.
        frames = frames.permute(0, 3, 1, 2)
        list_img = self._frame_to_list_img(frames)
        list_img = aug_transform(list_img)
        frames = self._list_img_to_frames(list_img)
        frames = frames.permute(0, 2, 3, 1)

        frames = utils.tensor_normalize(
            frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD
        )
        # T H W C -> C T H W.
        frames = frames.permute(3, 0, 1, 2)
        # Perform data augmentation.
        scl, asp = (
            self.cfg.DATA.TRAIN_JITTER_SCALES_RELATIVE,
            self.cfg.DATA.TRAIN_JITTER_ASPECT_RELATIVE,
        )
        relative_scales = (
            None if (self.mode not in ["train"] or len(scl) == 0) else scl
        )
        relative_aspect = (
            None if (self.mode not in ["train"] or len(asp) == 0) else asp
        )
        frames = utils.spatial_sampling(
            frames,
            spatial_idx=spatial_sample_index,
            min_scale=min_scale,
            max_scale=max_scale,
            crop_size=crop_size,
            random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
            inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
            aspect_ratio=relative_aspect,
            scale=relative_scales,
            motion_shift=self.cfg.DATA.TRAIN_JITTER_MOTION_SHIFT
            if self.mode in ["train"]
            else False,
        )

        if self.rand_erase:
            erase_transform = RandomErasing(
                self.cfg.AUG.RE_PROB,
                mode=self.cfg.AUG.RE_MODE,
                max_count=self.cfg.AUG.RE_COUNT,
                num_splits=self.cfg.AUG.RE_COUNT,
                device="cpu",
            )
            frames = frames.permute(1, 0, 2, 3)
            frames = erase_transform(frames)
            frames = frames.permute(1, 0, 2, 3)

        return frames

    def _frame_to_list_img(self, frames):
        img_list = [
            transforms.ToPILImage()(frames[i]) for i in range(frames.size(0))
        ]
        return img_list

    def _list_img_to_frames(self, img_list):
        img_list = [transforms.ToTensor()(img) for img in img_list]
        return torch.stack(img_list)

    def __len__(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return self.num_videos

    @property
    def num_videos(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return len(self._path_to_videos)
